// LPSEthTest.c : This file contains the 'main' function. Program execution begins and ends there.
//
// This version is an example CLI for Linux
#include <stdbool.h>
#include <stdio.h>
#include <stdint.h>         
#include <stdlib.h>
#include <Windows.h>
#include <math.h>
#include <string.h>
#include <ctype.h>

#include "stdafx.h"
#include "lpsdrvr.h"

#define FALSE 0
#define TRUE !FALSE

const char* lpsdevices[] = { "192.168.100.10" };
#define DEBUG_LEVEL 1
#define LPS_DEVICES  (sizeof(lpsdevices)/sizeof(lpsdevices[0]))

// IP Address Validator
#define DELIM "."

// ------------------------------ - Equates-----------------------------------------------
#define CL_SWP_DIRECTION		0x04	// MASK: bit = 0 for ramp up, 1 for ramp down 
#define CL_SWP_CONTINUOUS		0x02	// MASK: bit = 1 for continuous ramping
#define CL_SWP_ONCE				0x01	// MASK: bit = 1 for single ramp
#define CL_SWP_BIDIRECTIONALLY	0x10	// MASK: bit = 1 for bidirectional ramps

// --------------------------- Variables -------------------------------------------------
static DEVID MyDevices[MAXDEVICES];			// I have statically allocated this array for convenience
// It holds a list of device IDs for the connected devices
// They are stored starting at MyDevices[0]

static char* ndeviceip;
static int IdleTime = 1;		// default idle time is 1 ms
static int HoldTime = 1;		// default hold time is 1 ms
static int AStart = 0;		// default phase angle start level is 0 degrees.
static int AStop = 360;		// default phase angle stop level is 360 degrees.
static int Dwell = 1000;		// default dwell time is 1 second for first ramp phase
static int Dwell2 = 1000;		// default dwell time is 1 second for second ramp phase
static int AStep = 1;			// default step size is 1 degree
static int AStep2 = 1;		// default second phase step size for LPS's that support bidirectional ramps

static int WorkingFrequency = 0;	// working frequency
static int Phase = 0;			// default phase is 0 degrees

static int SerialNumber = 0;		// used to hold the serial number for the get serial number command

static int RFOnOff = 1;				// really used as a bool -- if non zero, turn on the RF output

static int Sweep_mode = 0;			// used to control the sweep mode
static int GetParam = 0;			// the low byte is the GET command byte

static int ProfileIndex = 0;		// the element in the profile we want to set
static int ProfileLength = 0;		// the length of the profile
static int ProfileValue = 0;		// the profile element's value

static int Channel = 1;				// we just default to the first channel
static int NumToRamp = 0;			// number of channels we should ramp (also used as an index -- post increment!)
static int ChannelList[8] = { 0, 0, 0, 0, 0, 0, 0, 0 };
static int iRampMode = 0;
static char glpsdevicesip[16];       // devices ip address

bool gbDeviceOpen = FALSE;
bool gbWantSetIdle = FALSE;
bool gbWantSetHold = FALSE;
bool gbWantSetAStart = FALSE;
bool gbWantSetAStop = FALSE;
bool gbWantSetDwell = FALSE;
bool gbWantSetDwell2 = FALSE;
bool gbWantStartSweep = FALSE;
bool gbWantSetAStep = FALSE;
bool gbWantSetAStep2 = FALSE;
bool gbWantSetWorkingFrequency = FALSE;
bool gbWantSetPhase = FALSE;
bool gbWantSaveSettings = FALSE;
bool gbWantGetParam = FALSE;
bool gbBatchMode = FALSE;
bool gbWantSetRFOnOff = FALSE;
bool gbQuietMode = FALSE;
bool gbWantSetProfileElement = FALSE;
bool gbWantSetProfileLength = FALSE;
bool gbWantChannel = FALSE;
bool gbWantMCRamp = FALSE;


//***************************************************************************
//
// Validate netconfig digits and return 1 if string contain only digits, else return 0
//
//*****************************************************************************
static int valid_digit(char* ip_str) {
	while (*ip_str) {
		if (*ip_str >= '0' && *ip_str <= '9')
			++ip_str;
		else
			return 0;
	}
	return 1;
}

//***************************************************************************
//
// Validate IP Configuration -return 1 if IP string is valid, else return 0
//
//*****************************************************************************
static int is_valid_ip(char* ip_str) {
	int num, dots = 0;
	char* ptr;
	char lstr[16];
	char* next_token;
	strcpy_s(lstr, sizeof(lstr), ip_str);

	if (lstr == NULL)
		return 0;

	ptr = strtok_s(lstr, DELIM, &next_token);

	if (ptr == NULL)
		return 0;

	while (ptr) {

		/* after parsing string, it must contain only digits */
		if (!valid_digit(ptr))
			return 0;

		num = atoi(ptr);

		/* check for valid IP */
		if (num >= 0 && num <= 255) {
			/* parse remaining string */
			ptr = strtok_s(NULL, DELIM, &next_token);
			if (ptr != NULL)
				++dots;
		}
		else
			return 0;
	}

	/* valid IP string must contain 3 dots */
	if (dots != 3)
		return 0;
	return 1;
}

int ChannelToMask(int channel)
{
	int mask = 1;
	if (channel < 1) channel = Channel;		// channel 0 signifies the current global channel
	return mask << (channel - 1);			// channel runs from 1 to 8
}

void PrintHelp()
{
	printf(" --- Overall modes and device selection based on ip address ---\n");
	printf("  -d i 		Select the device to work with, i is the device ip address (192.168.100.11)\n");
	printf("  -r  		Read the current information of the device\n");
	printf("  -y		Save the current settings in the device.\n");
	printf("\n");
	printf("  -b		Batch mode, exit immediately after sending commands to the Lab Bricks.\n");
	printf("  -q		Quiet mode, skip most outputs.\n");
	printf("\n");

	printf(" --- Commands to set parameters and start ramp --- \n");
	printf("  -c n      Set the active channel\n");
	printf("  -f nn     Set working frequency, nn is working frequency in MHz\n");
	printf("  -a nn     Set phase angle, nn is phase in degrees\n");
	printf("  -w nn     Set idle time between phase ramps, nn is time in ms.\n");
	printf("  -h nn     Set hold time between ramp phases\n");
	printf("  -s nn     Set ramp start value, nn is start value in degrees\n");
	printf("  -e nn     Set ramp end value, nn is end value in degrees\n");
	printf("  -t p nn   Set time to dwell on each phase value, nn is time in ms., p is ramp phase 1 or 2\n");

	printf("  -i p nn   Set phase ramp increment, nn is the increment\n");
	printf("            in degrees. p is ramp phase 1 or 2\n");
	printf("  -g n      Start a ramp, 1 = once upwards, 2 = continuous upwards\n");
	printf("            5 = once down, 6 = continuous down, 17 = bidirectional once,\n");
	printf("            18 = continuous bidirectional ramps, 0 to stop\n");

	printf("  -x m n c c c ...	Start multiple ramps, m is the mode: \n");
	printf("					1 = once upwards, 2 = continuous upwards\n");
	printf("					5 = once down, 6 = continuous down, 17 = bidirectional once,\n");
	printf("					18 = continuous bidirectional ramps, 0 to stop,\n");
	printf("					n = number of profiles, for each one c is the channel number\n");

	printf("\n");


}

// -------------------- - MakeLower------------------------------

static void MakeStrLower(char* str)
{
	int i;

	for (i = 0; i < strlen(str); i++)
	{
		str[i] = (char)tolower(str[i]);
	}
}

// remove the first character from the string
static void RemoveDash(char* str)
{
	int i;

	for (i = 0; i < strlen(str); i++)
	{
		str[i] = str[i + 1];				// in effect we just shift the chars by one position
	}
}


// ParseCommandLine() will return FALSE to indicate that we received an invalid
// command or should abort for another reason.
bool ParseCommandLine(int argc, char* argv[])
{
	int RampPhase;
	char devipstr[16];
	char thisParam[256];
	size_t icount;
	int i;
	int iActiveChannel = 0;

	enum {
		wantDash, wantIdle, wantAStart, wantAStop, wantDwell, wantAStep,
		wantPhase, wantSetRFOnOff, wantSweep, wantGetParam, wantDevID,
		wantDwell2, wantAStep2, wantHold, wantDwellPhase, wantStepPhase, wantWorkingFrequency,
		wantChannel, wantRampMode, wantNumToRamp, wantRampCmd
	} state = wantDash;

	thisParam[255] = 0;	// safety backstop for errant strings


	for (i = 1; i < argc; ++i) {

		// Convert each argument to lowercase
		strncpy_s(thisParam, argv[i], 255);		// the limit is just to defend our string arrays

		MakeStrLower(thisParam);

		if (state == wantDash)
		{
			if ('-' != thisParam[0])
			{
				printf("\n *** Error in command line syntax *** \n");
				PrintHelp();
				return FALSE;
			}
			// remove the dash from the front of the string
			RemoveDash(thisParam);

			// Identify the command line arguments
			if ('d' == thisParam[0]) {
				state = wantDevID;
			}
			else if ('r' == thisParam[0]) {
				gbWantGetParam = TRUE;
				state = wantDash;
			}
			else if ('w' == thisParam[0]) {
				gbWantSetIdle = TRUE;
				state = wantIdle;
			}
			else if ('s' == thisParam[0]) {
				gbWantSetAStart = TRUE;
				state = wantAStart;
			}
			else if ('e' == thisParam[0]) {
				gbWantSetAStop = TRUE;
				state = wantAStop;
			}
			else if ('t' == thisParam[0]) {
				state = wantDwellPhase;
			}
			else if ('i' == thisParam[0]) {
				state = wantStepPhase;
			}
			else if ('a' == thisParam[0]) {
				gbWantSetPhase = TRUE;
				state = wantPhase;
			}
			else if ('g' == thisParam[0]) {
				gbWantStartSweep = TRUE;
				state = wantSweep;
			}
			else if ('r' == thisParam[0]) {
				gbWantSetRFOnOff = TRUE;
				state = wantSetRFOnOff;
			}
			else if ('y' == thisParam[0]) {
				gbWantSaveSettings = TRUE;
				state = wantDash;
			}
			else if ('b' == thisParam[0]) {
				gbBatchMode = TRUE;
				state = wantDash;
			}
			else if ('q' == thisParam[0]) {
				gbQuietMode = TRUE;
				state = wantDash;
			}
			else if ('h' == thisParam[0]) {
				gbWantSetHold = TRUE;
				state = wantHold;
			}
			else if ('f' == thisParam[0]) {
				gbWantSetWorkingFrequency = TRUE;
				state = wantWorkingFrequency;
			}
			else if ('c' == thisParam[0]) {
				gbWantChannel = TRUE;
				state = wantChannel;
			}
			else if ('x' == thisParam[0]) {
				gbWantMCRamp = TRUE;
				NumToRamp = 0;
				iRampMode = 0;
				state = wantRampMode;
			}

			else {
				// this case is for "-?" and any argument we don't recognize
				PrintHelp();
				return FALSE;	// don't continue
			}
		}

		else {

			// save the whole substring and do conversions for each argument type
			switch (state) {

			case wantDevID:
				if (is_valid_ip(thisParam))
				{
					strcpy_s(glpsdevicesip, sizeof(glpsdevicesip), thisParam);
					gbDeviceOpen = TRUE;
				}
				else
					printf("Invalid IP Address\n");
				state = wantDash;
				break;

			case wantDwellPhase:
				RampPhase = (int)atoi(thisParam);
				if (RampPhase == 1) {
					gbWantSetDwell = TRUE;
					state = wantDwell;
				}
				else if (RampPhase == 2) {
					gbWantSetDwell2 = TRUE;
					state = wantDwell2;
				}
				else state = wantDash;		// phase value is wrong, not much we can do about it...
				break;

			case wantStepPhase:
				RampPhase = (int)atoi(thisParam);
				if (RampPhase == 1) {
					gbWantSetAStep = TRUE;
					state = wantAStep;
				}
				else if (RampPhase == 2) {
					gbWantSetAStep2 = TRUE;
					state = wantAStep2;
				}
				else state = wantDash;		// phase value is wrong, not much we can do about it...
				break;

			case wantIdle:
				IdleTime = (int)atoi(thisParam);	// convert to a int
				state = wantDash;
				break;

			case wantHold:
				HoldTime = (int)atoi(thisParam);
				state = wantDash;
				break;

			case wantChannel:
				Channel = (int)atoi(thisParam);
				state = wantDash;
				break;

			case wantAStart:
				AStart = (int)atof(thisParam);
				state = wantDash;
				break;

			case wantAStop:
				AStop = (int)atof(thisParam);
				state = wantDash;
				break;

			case wantDwell:
				Dwell = (int)atoi(thisParam);
				state = wantDash;
				break;

			case wantDwell2:
				Dwell2 = (int)atoi(thisParam);
				state = wantDash;
				break;

			case wantAStep:
				AStep = (int)atof(thisParam);
				state = wantDash;
				break;

			case wantAStep2:
				AStep2 = (int)atof(thisParam);
				state = wantDash;
				break;

			case wantPhase:
				Phase = (int)atof(thisParam);	// cast to a float, atof actually returns a double
				state = wantDash;
				break;

			case wantWorkingFrequency:
				WorkingFrequency = (int)atof(thisParam); // cast to an int, atof actually returns a double
				state = wantDash;
				break;

			case wantSetRFOnOff:
				RFOnOff = (int)atoi(thisParam);
				state = wantDash;
				break;

			case wantSweep:
				Sweep_mode = (int)atoi(thisParam);
				state = wantDash;
				break;

			case wantRampMode:
				iRampMode = (int)atoi(thisParam);
				state = wantNumToRamp;
				break;

			case wantNumToRamp:
				NumToRamp = (int)atoi(thisParam);
				if (NumToRamp < 0) NumToRamp = 0;
				if (NumToRamp > 8) NumToRamp = 8;
				state = wantRampCmd;
				break;

			case wantRampCmd:
				ChannelList[iActiveChannel] = (int)atoi(thisParam);
				iActiveChannel++;
				if (iActiveChannel < NumToRamp) state = wantRampCmd;
				else state = wantDash;
				break;

			}
		}
	}

	if (state != wantDash) {
		// we are expecting an argument, if we didn't get one then print the help message
		PrintHelp();
		return FALSE;
	}

	// It's OK to continue
	return TRUE;
}


int main(int argc, char* argv[])
{
	bool realhardware;
	unsigned int index = 0;
	int idata;
	float fdata;
	bool bTemp;
	int devid;

	printf("Vaunix Phase Shifter Demonstration\n");
	int version = fnLPS_GetLibVersion();
	int major = (version >> 8) & 0xFF;
	int minor = version & 0xFF;
	printf("Library Version %d.%d\n", major, minor);
	printf("\n");
	printf("Hit CTRL+C to exit\n");
	printf("\n");

	if (!ParseCommandLine(argc, argv))
		return 0;


	/* If you have actual hardware attached, set this to TRUE. Setting to FALSE will run in test mode */
	realhardware = TRUE;
	fnLPS_SetTestMode(!realhardware);

	// Add the LPS Device
	if (gbDeviceOpen)
	{
		fnLPS_AddLPSDevice(glpsdevicesip);
	}

	idata = fnLPS_GetNumDevices();

	if (idata == 0) {
		printf("No device found\n");
	}
	else if (idata < 0)
	{
		printf("GetNumDevices Returned %s\n", fnLPS_perror(idata));		// an example of displaying API error returns
	}

	if (idata == 1) {
		if (!gbQuietMode) printf("Found %d Device\n", idata);
	}
	else {
		if (!gbQuietMode) printf("Found %d Devices\n", idata);
	}

	idata = fnLPS_GetDevInfo(MyDevices);

	if (idata < 0)
	{
		printf("GetDevInfo Returned %s\n", fnLPS_perror(idata));		// an example of displaying API error returns
	}

	if (!gbQuietMode) printf("Got Device Info for %d Device[s]\n", idata);

	// Get the device ID of the requested device
	devid = MyDevices[0];

	// Initialize the device
	idata = fnLPS_InitDevice(devid);

	if (idata < 0)
	{
		printf("InitDevice Returned %s\n", fnLPS_perror(idata));
		return 0;
	}

	int numChannels = fnLPS_GetNumChannels(devid);

	// Require channel for setting parameters
	if (numChannels > 1) {
		if (!gbWantChannel && (gbWantSetIdle || gbWantSetHold || gbWantSetAStart || gbWantSetAStop || gbWantSetDwell ||
			gbWantSetDwell2 || gbWantStartSweep || gbWantSetAStep || gbWantSetAStep2 || gbWantSetWorkingFrequency ||
			gbWantSetPhase || gbWantSaveSettings || gbWantSetRFOnOff || gbWantSetProfileElement || gbWantSetProfileLength)
			) {
			printf("Multiple channels detected. Please specify a channel.\n");
			return 0;
		}
	}

	if (gbWantGetParam)
	{
		if (fnLPS_CheckDeviceReady(devid) == STATUS_OK)
		{
			printf("--------- Device %d -------------------------\n", devid);

			// Device Name
			char getmodelname[MAX_MODELNAME];
			memset(getmodelname, 0, sizeof(getmodelname));
			fnLPS_GetModelName(devid, getmodelname);
			printf("Model Name: %s\n", getmodelname);

			// Serial Number
			idata = fnLPS_GetSerialNumber(devid);
			printf("Serial Number: %d\n", idata);

			// IP Mode
			idata = -1;
			fnLPS_GetIPMode(devid, &idata);
			printf("IP Mode: %d\n", idata);

			// IP Address
			char getipaddress[32];
			memset(getipaddress, 0, sizeof(getipaddress));
			fnLPS_GetIPAddress(devid, getipaddress);
			printf("IP Address: %s\n", getipaddress);

			// Netmask
			char getnetmask[32];
			memset(getnetmask, 0, sizeof(getnetmask));
			fnLPS_GetNetmask(devid, getnetmask);
			printf("Subnet Mask: %s\n", getnetmask);

			// Gateway
			char getgateway[32];
			memset(getgateway, 0, sizeof(getgateway));
			fnLPS_GetGateway(devid, getgateway);
			printf("Gateway: %s\n", getgateway);

			// Min Frequency
			idata = fnLPS_GetMinWorkingFrequency(devid);
			printf("Min Frequency: %d MHz\n", (idata / 10));

			// Max Frequency
			idata = fnLPS_GetMaxWorkingFrequency(devid);
			printf("Max Frequency: %d MHz\n", (idata / 10));

			// Min Phase Shift
			idata = fnLPS_GetMinPhaseShift(devid);
			printf("Min Phase Shift: %d degrees\n", idata);

			// Max Phase Shift
			idata = fnLPS_GetMaxPhaseShift(devid);
			printf("Max Phase Shift: %d degrees\n", idata);

			for (int i = 0; i < numChannels; i++)
			{
				fnLPS_SetChannel(devid, i + 1);		// set the channel so we can read its parameters
				printf("--------- Channel %d/%d ------------\n", i + 1, numChannels);

				// Device Status
				idata = fnLPS_GetDeviceStatus(devid);
				printf("Status: ");
				for (int i = sizeof(idata) * 8 - 1; i >= 0; i--) {
					printf("%d", (idata >> i) & 1);
				}
				printf("\n");

				// Current Frequency
				idata = fnLPS_GetWorkingFrequency(devid);
				printf("Frequency: %d MHz\n", (idata / 10));

				// Current Phase
				idata = fnLPS_GetPhaseAngle(devid);
				printf("Phase Angle: %d degrees\n", idata);

				// Ramp Start
				idata = fnLPS_GetRampStart(devid);
				printf("Ramp Start: %d degrees\n", idata);

				// Ramp End
				idata = fnLPS_GetRampEnd(devid);
				printf("Ramp Stop: %d degrees\n", idata);

				// Dwell Time
				idata = fnLPS_GetDwellTime(devid);
				printf("Dwell Time: %d ms\n", idata);

				// Dwell Time2
				idata = fnLPS_GetDwellTimeTwo(devid);
				printf("Bi-directional Dwell Time: %d ms\n", idata);

				// Idle Time
				idata = fnLPS_GetIdleTime(devid);
				printf("Idle Time: %d ms\n", idata);

				// Hold Time
				idata = fnLPS_GetHoldTime(devid);
				printf("Hold Time: %d ms\n", idata);

				// Get Profile Count
				idata = fnLPS_GetProfileCount(devid);
				printf("Profile Count: %d\n", idata);

				// Get Profile Max Length
				idata = fnLPS_GetProfileMaxLength(devid);
				printf("Profile Max Length: %d\n", idata);

				// Get Profile Dwell Time
				idata = fnLPS_GetProfileDwellTime(devid);
				printf("Profile Dwell Time: %d ms\n", idata);

				// Get Profile Idle Time
				idata = fnLPS_GetProfileIdleTime(devid);
				printf("Profile Idle Time: %d ms\n", idata);

				// Get Profile Index
				idata = fnLPS_GetProfileIndex(devid);
				printf("Profile Index: %d\n", idata);
			}

		}
		else {
			printf("Device %d not ready\n", devid);
			return 0;
		}
		gbWantGetParam = FALSE;
	}

	//Set channel
	if (gbWantChannel)
	{
		if (!gbQuietMode) printf("Setting the channel for device %d to %d\n", devid, Channel);
		fnLPS_SetChannel(devid, Channel);
	}

	// fnLPS_SetProfileElement(devid, 0, 10);
	// fnLPS_SetProfileElement(devid, 1, 20);
	// fnLPS_SetProfileElement(devid, 2, 30);
	// fnLPS_SetProfileElement(devid, 3, 40);
	// fnLPS_SetProfileElement(devid, 4, 50);
	// fnLPS_SetProfileCount(devid, 5);
	// fnLPS_SetProfileIdleTime(devid, 100);
	// fnLPS_SetProfileDwellTime(devid, 200);
	// fnLPS_StartProfile(devid, PROFILE_REPEAT);

	// Set Frequency
	if (gbWantSetWorkingFrequency)
	{
		if (!gbQuietMode) printf("Working Frequency set to  = %d Mhz\n", WorkingFrequency);
		fnLPS_SetWorkingFrequency(devid, WorkingFrequency * 10);
	}

	// Set Phase
	if (gbWantSetPhase)
	{
		// Set the selected channel with the phase
		if (!gbQuietMode) printf("Setting the phase for device %d to %d degrees\n", devid, Phase);
		fnLPS_SetPhaseAngle(devid, Phase);
	}


	// Idle time
	if (gbWantSetIdle)
	{
		if (!gbQuietMode) printf("Setting the idle time between ramps for device %d to %d ms. \n", devid, IdleTime);
		fnLPS_SetIdleTime(devid, IdleTime);
	}

	// Hold time
	if (gbWantSetHold)
	{
		if (!gbQuietMode) printf("Setting the hold time between ramp phases for device %d to %d ms \n", devid, HoldTime);
		fnLPS_SetHoldTime(devid, HoldTime);
	}

	// DWell time
	if (gbWantSetDwell) {

		if (!gbQuietMode) printf("Setting the first phase dwell time for device %d to %d \n", devid, Dwell);
		fnLPS_SetDwellTime(devid, Dwell);
	}

	// Dwell time2
	if (gbWantSetDwell2)
	{
		if (!gbQuietMode) printf("Setting the second phase dwell time for device %d to %d ms \n", devid, Dwell2);
		fnLPS_SetDwellTimeTwo(devid, Dwell2);
	}

	// Ramp Start
	if (gbWantSetAStart) {

		if (!gbQuietMode) printf("Setting the ramp start for device %d to %d degrees \n", devid, AStart);
		fnLPS_SetRampStart(devid, AStart);
	}

	// Ramp Stop
	if (gbWantSetAStop) {

		if (!gbQuietMode) printf("Setting ramp end for device %d to %d degrees \n", devid, AStop);
		fnLPS_SetRampEnd(devid, AStop);
	}


	if (gbWantSetAStep) {
		if (!gbQuietMode) printf("Setting the first phase angle step for device %d to %d degrees \n", devid, AStep);
		fnLPS_SetPhaseAngleStep(devid, AStep);
	}


	if (gbWantSetAStep2) {
		if (!gbQuietMode) printf("Setting the second phase angle step2 for device %d to %d degrees \n", devid, AStep2);
		fnLPS_SetPhaseAngleStepTwo(devid, AStep2);
	}

	// Ramp Mode and Direction
	if (gbWantStartSweep)
	{
		if (Sweep_mode == 0)
		{
			if (!gbQuietMode) printf("Stopping the Phase Angle Ramp\n");
			fnLPS_StartRamp(devid, FALSE);

		}
		else
		{

			// --- The user wants to start some kind of a phase angle ramp ---
			if (Sweep_mode & CL_SWP_DIRECTION)
			{
				bTemp = FALSE;
			}
			else
			{
				bTemp = TRUE;
			}	// NB -- the flag is TRUE for "up" in the Set...Direction call.
			// but the old test program uses a 0 bit for up, and a 1 bit for down...

			fnLPS_SetRampDirection(devid, bTemp);

			// --- and now we'll do the mode - one time or repeated ---
			if (Sweep_mode & CL_SWP_ONCE)
			{
				bTemp = FALSE;
			}
			else
			{
				bTemp = TRUE;
			}	// NB -- the flag is TRUE for "repeated" in the SetSweepMode call.
			// but the old test program encodes the modes differently

			fnLPS_SetRampMode(devid, bTemp);

			if (Sweep_mode & CL_SWP_BIDIRECTIONALLY)
			{
				bTemp = TRUE;
			}							// the command line has true for bidirectional 
			else						// as does the actual HW command...
			{
				bTemp = FALSE;
			}

			printf("Bidirection mode set to %x \n", bTemp);
			fnLPS_SetRampBidirectional(devid, bTemp);

			if (!gbQuietMode) printf("Starting a phase angle ramp for device %d\n", devid);
			printf("%d\n", fnLPS_StartRamp(devid, TRUE));

		}
	}

	if (gbWantMCRamp)
	{
		// --- first we'll get the number of channels for this device
		int NumChannels = fnLPS_GetNumChannels(devid);
		int ChMask = 0;

		if ((NumChannels <= 0) || (NumChannels > 8)) NumChannels = 1;	// defend against bad values

		// --- then we loop over the selected channels, generating a channel mask from our active channel list
		for (int i = 0; i < NumToRamp; i++) {
			ChMask |= ChannelToMask(ChannelList[i]);		// add a bit to our mask for each active channel
		}

		// --- launch the set of selected ramps for this device ---
		if (!gbQuietMode) printf("Starting ramps for device %d\n", devid);
		printf("ChMask (binary): ");
		for (int bit = 7; bit >= 0; bit--) {
			printf("%d", (ChMask >> bit) & 1);
		}
		fnLPS_StartRampMC(devid, iRampMode, ChMask, FALSE);		// start the ramps immediately
	}

	// Save settings
	if (gbWantSaveSettings)
	{
		if (!gbQuietMode) printf("Saving the settings for device %d\n", devid);
		fnLPS_SaveSettings(devid);
	}

	int itemp = -1;
	int NumChannels = fnLPS_GetNumChannels(devid);
	printf("\nReading phase angle for device %d on %d channels...\n", devid, NumChannels);
	for (int i = 0; i < 10; ++i) {
		if (NumChannels <= 0) NumChannels = 1;	// protect against an error return
		for (int k = 1; k < NumChannels + 1; k++)
		{
			fnLPS_SetChannel(devid, k);
			itemp = fnLPS_GetPhaseAngle(devid);
			printf("Phase Angle = %d degrees for device %d, channel %d\n", itemp, devid, k);
		}
		printf("\n");
		Sleep(200);
	}

	// Close the device socket at the end of the process
	if (gbDeviceOpen)
	{
		fnLPS_CloseDevice(devid);
		gbDeviceOpen = FALSE;
	}

}
